<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class WhatsAppService
{
    protected $apiUrl;
    protected $name;
    protected $token;
    protected $timeout;
    protected $retryAttempts;

    public function __construct()
    {
        $this->apiUrl = config('whatsapp.api_url');
        $this->name = config('whatsapp.name');
        $this->token = config('whatsapp.token');
        $this->timeout = config('whatsapp.timeout', 30);
        $this->retryAttempts = config('whatsapp.retry_attempts', 3);
    }

    /**
     * Enviar mensagem de texto simples
     *
     * @param string $number Número com DDD (ex: 27992494563)
     * @param string $message Mensagem a ser enviada
     * @param array $options Opções adicionais
     * @return array
     */
    public function sendText($number, $message, $options = [])
    {
        // Validar número
        $number = $this->formatNumber($number);
        
        if (!$number) {
            return [
                'success' => false,
                'message' => 'Número inválido'
            ];
        }

        // Verificar se está em rate limit
        $cacheKey = "whatsapp_rate_limit_{$number}";
        if (Cache::has($cacheKey)) {
            return [
                'success' => false,
                'message' => 'Rate limit: Aguarde alguns segundos antes de enviar novamente'
            ];
        }

        // Preparar URL
        $url = rtrim($this->apiUrl, '/') . '/';
        $params = [
            'name' => $this->name,
            'token' => $this->token,
            'number' => $number,
            'message' => $message,
        ];

        // Tentar enviar com retry
        $attempt = 0;
        $lastError = null;

        while ($attempt < $this->retryAttempts) {
            try {
                $response = Http::timeout($this->timeout)
                    ->get($url, $params);

                if ($response->successful()) {
                    // Aplicar rate limit (1 mensagem por segundo por número)
                    Cache::put($cacheKey, true, 1);

                    // Log de sucesso
                    Log::info('WhatsApp message sent', [
                        'number' => $number,
                        'message_length' => strlen($message),
                        'attempt' => $attempt + 1
                    ]);

                    return [
                        'success' => true,
                        'message' => 'Mensagem enviada com sucesso',
                        'response' => $response->body()
                    ];
                } else {
                    $lastError = "HTTP {$response->status()}: {$response->body()}";
                }
            } catch (\Exception $e) {
                $lastError = $e->getMessage();
                Log::warning('WhatsApp send attempt failed', [
                    'number' => $number,
                    'attempt' => $attempt + 1,
                    'error' => $lastError
                ]);
            }

            $attempt++;
            if ($attempt < $this->retryAttempts) {
                sleep(config('whatsapp.retry_delay', 5));
            }
        }

        // Log de erro final
        Log::error('WhatsApp message failed after retries', [
            'number' => $number,
            'error' => $lastError
        ]);

        return [
            'success' => false,
            'message' => 'Erro ao enviar mensagem: ' . $lastError
        ];
    }

    /**
     * Enviar notificação de nova notícia
     *
     * @param string $number
     * @param string $title
     * @param string $url
     * @param string $type Tipo: new_post, breaking_news, scheduled_post
     * @return array
     */
    public function sendPostNotification($number, $title, $url, $type = 'new_post')
    {
        $template = config("whatsapp.templates.{$type}", config('whatsapp.templates.new_post'));
        
        $message = str_replace(
            ['{title}', '{url}'],
            [$title, $url],
            $template
        );

        return $this->sendText($number, $message);
    }

    /**
     * Enviar para múltiplos números (em fila)
     *
     * @param array $numbers
     * @param string $message
     * @return array
     */
    public function sendBulk(array $numbers, string $message)
    {
        $results = [];
        
        foreach ($numbers as $number) {
            $results[$number] = $this->sendText($number, $message);
            // Pequeno delay entre envios
            usleep(500000); // 0.5 segundos
        }

        return $results;
    }

    /**
     * Formatar número de telefone
     * Formato esperado: DDD+Número (ex: 27992494563)
     *
     * @param string $number
     * @return string|null
     */
    protected function formatNumber($number)
    {
        // Remover caracteres não numéricos
        $number = preg_replace('/[^0-9]/', '', $number);

        // Se começar com 0, remover
        if (substr($number, 0, 1) === '0') {
            $number = substr($number, 1);
        }

        // Se começar com +55 ou 55, remover (API espera apenas DDD+Número)
        if (substr($number, 0, 2) === '55') {
            $number = substr($number, 2);
        }

        // Validar tamanho (mínimo 10 dígitos: DDD + número)
        // DDD (2 dígitos) + número (8 ou 9 dígitos)
        if (strlen($number) < 10 || strlen($number) > 11) {
            return null;
        }

        return $number;
    }

    /**
     * Verificar se o serviço está configurado
     *
     * @return bool
     */
    public function isConfigured()
    {
        return !empty($this->name) && !empty($this->token);
    }
}

