<?php
/**
 * Script de Instalação do Portal Bahia
 * Baixa arquivos do servidor central e instala automaticamente
 */

header('Content-Type: text/event-stream');
header('Cache-Control: no-cache');
header('Connection: keep-alive');

// Desabilitar time limit
set_time_limit(0);
ignore_user_abort(true);

function sendProgress($progress, $status = '', $log = '') {
    if ($progress > 0) {
        echo "progress: $progress\n";
    }
    if ($status) {
        echo "status: $status\n";
    }
    if ($log) {
        echo "log: $log\n";
    }
    echo "\n";
    ob_flush();
    flush();
}

// Verificar se é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die('Method not allowed');
}

$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? '';
$config = $input['config'] ?? [];

if (empty($config)) {
    sendProgress(0, 'Erro: Configuração não fornecida', '❌ Dados de configuração não recebidos');
    exit;
}

$rootDir = dirname(__DIR__);
$serverUrl = rtrim($config['server_url'] ?? '', '/');
$masterUrl = rtrim($config['master_url'] ?? '', '/');
$licenseKey = $config['license_key'] ?? '';

if ($action === 'download') {
    sendProgress(0, 'Iniciando download...', '📥 Conectando ao servidor central...');
    
    // Validar licença primeiro
    sendProgress(5, 'Validando licença...', '🔐 Validando licença com Master Panel...');
    
    $validationUrl = $masterUrl . '/api/v1/validate-license.php?license_key=' . urlencode($licenseKey);
    $ch = curl_init($validationUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        sendProgress(0, 'Erro na validação', '❌ Erro ao validar licença. HTTP Code: ' . $httpCode);
        exit;
    }
    
    $result = json_decode($response, true);
    if (!isset($result['success']) || !$result['success']) {
        sendProgress(0, 'Licença inválida', '❌ ' . ($result['message'] ?? 'Licença inválida ou expirada'));
        exit;
    }
    
    sendProgress(10, 'Licença válida', '✅ Licença validada com sucesso');
    
    // Baixar arquivos do servidor central
    sendProgress(15, 'Baixando arquivos...', '📦 Baixando pacote do servidor central...');
    
    // URL do download: servidor.canal33.com.br/download/portal-bahia.zip
    $downloadUrl = rtrim($serverUrl, '/') . '/download/portal-bahia.zip';
    $zipFile = $rootDir . '/portal-bahia-temp.zip';
    
    $ch = curl_init($downloadUrl);
    $fp = fopen($zipFile, 'w');
    curl_setopt($ch, CURLOPT_FILE, $fp);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_PROGRESSFUNCTION, function($resource, $download_size, $downloaded, $upload_size, $uploaded) {
        if ($download_size > 0) {
            $progress = 15 + (($downloaded / $download_size) * 70);
            sendProgress((int)$progress, 'Baixando arquivos...', "📥 " . round($downloaded / 1024 / 1024, 2) . " MB / " . round($download_size / 1024 / 1024, 2) . " MB");
        }
    });
    curl_setopt($ch, CURLOPT_NOPROGRESS, false);
    
    if (!curl_exec($ch)) {
        fclose($fp);
        unlink($zipFile);
        sendProgress(0, 'Erro no download', '❌ Erro ao baixar arquivos: ' . curl_error($ch));
        curl_close($ch);
        exit;
    }
    
    curl_close($ch);
    fclose($fp);
    
    sendProgress(85, 'Extraindo arquivos...', '📂 Extraindo arquivos...');
    
    // Extrair ZIP
    $zip = new ZipArchive();
    if ($zip->open($zipFile) === TRUE) {
        $zip->extractTo($rootDir);
        $zip->close();
        unlink($zipFile);
        sendProgress(100, 'Download concluído', '✅ Arquivos baixados e extraídos com sucesso');
    } else {
        unlink($zipFile);
        sendProgress(0, 'Erro na extração', '❌ Erro ao extrair arquivos');
        exit;
    }
    
} elseif ($action === 'install') {
    sendProgress(0, 'Iniciando instalação...', '🚀 Iniciando processo de instalação...');
    
    // 1. Criar .env
    sendProgress(5, 'Configurando ambiente...', '⚙️ Criando arquivo .env...');
    
    $envContent = "APP_NAME=\"Portal Bahia\"\n";
    $envContent .= "APP_ENV=production\n";
    $envContent .= "APP_KEY=\n";
    $envContent .= "APP_DEBUG=false\n";
    $envContent .= "APP_TIMEZONE=America/Sao_Paulo\n";
    $envContent .= "APP_URL=" . (isset($_SERVER['HTTPS']) ? 'https' : 'http') . "://" . $_SERVER['HTTP_HOST'] . "\n";
    $envContent .= "APP_LOCALE=pt_BR\n";
    $envContent .= "APP_FALLBACK_LOCALE=pt_BR\n\n";
    
    $envContent .= "DB_CONNECTION=mysql\n";
    $envContent .= "DB_HOST=" . $config['db_host'] . "\n";
    $envContent .= "DB_PORT=" . $config['db_port'] . "\n";
    $envContent .= "DB_DATABASE=" . $config['db_database'] . "\n";
    $envContent .= "DB_USERNAME=" . $config['db_username'] . "\n";
    $envContent .= "DB_PASSWORD=" . $config['db_password'] . "\n\n";
    
    $envContent .= "MASTER_PANEL_URL=" . $config['master_url'] . "\n";
    $envContent .= "MASTER_PANEL_LICENSE_KEY=" . $config['license_key'] . "\n\n";
    
    $envContent .= "CACHE_STORE=database\n";
    $envContent .= "SESSION_DRIVER=database\n";
    $envContent .= "QUEUE_CONNECTION=database\n";
    
    file_put_contents($rootDir . '/.env', $envContent);
    sendProgress(10, 'Arquivo .env criado', '✅ Arquivo .env criado');
    
    // 2. Gerar APP_KEY
    sendProgress(15, 'Gerando chave...', '🔑 Gerando APP_KEY...');
    $key = 'base64:' . base64_encode(random_bytes(32));
    $envContent = str_replace('APP_KEY=', 'APP_KEY=' . $key, file_get_contents($rootDir . '/.env'));
    file_put_contents($rootDir . '/.env', $envContent);
    sendProgress(20, 'Chave gerada', '✅ APP_KEY gerada');
    
    // 3. Instalar dependências
    sendProgress(25, 'Instalando dependências...', '📦 Executando composer install...');
    
    $composerPath = null;
    $paths = ['composer', '/usr/local/bin/composer', '/usr/bin/composer'];
    foreach ($paths as $path) {
        if ($path === 'composer') {
            $output = [];
            exec('composer --version 2>&1', $output, $return);
            if ($return === 0) {
                $composerPath = 'composer';
                break;
            }
        } elseif (file_exists($path)) {
            $composerPath = $path;
            break;
        }
    }
    
    if (!$composerPath) {
        sendProgress(0, 'Composer não encontrado', '❌ Composer não encontrado. Instale o Composer primeiro.');
        exit;
    }
    
    $command = $composerPath . ' install --no-dev --optimize-autoloader --no-interaction 2>&1';
    exec($command, $output, $return);
    
    if ($return !== 0) {
        sendProgress(0, 'Erro no Composer', '❌ Erro ao instalar dependências: ' . implode("\n", $output));
        exit;
    }
    
    sendProgress(40, 'Dependências instaladas', '✅ Dependências instaladas');
    
    // 4. Criar banco de dados (se não existir)
    sendProgress(45, 'Configurando banco...', '🗄️ Conectando ao banco de dados...');
    
    try {
        $pdo = new PDO(
            "mysql:host={$config['db_host']};port={$config['db_port']}",
            $config['db_username'],
            $config['db_password']
        );
        
        // Tentar criar banco se não existir
        $pdo->exec("CREATE DATABASE IF NOT EXISTS `{$config['db_database']}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
        sendProgress(50, 'Banco configurado', '✅ Banco de dados configurado');
    } catch (PDOException $e) {
        sendProgress(0, 'Erro no banco', '❌ Erro ao conectar ao banco: ' . $e->getMessage());
        exit;
    }
    
    // 5. Executar migrations
    sendProgress(55, 'Criando tabelas...', '📊 Executando migrations...');
    
    $command = "cd " . escapeshellarg($rootDir) . " && php artisan migrate --force 2>&1";
    exec($command, $output, $return);
    
    if ($return !== 0) {
        sendProgress(0, 'Erro nas migrations', '❌ Erro ao executar migrations: ' . implode("\n", array_slice($output, -10)));
        exit;
    }
    
    sendProgress(70, 'Tabelas criadas', '✅ Tabelas criadas com sucesso');
    
    // 6. Executar seeders
    sendProgress(75, 'Carregando dados iniciais...', '🌱 Executando seeders...');
    
    $command = "cd " . escapeshellarg($rootDir) . " && php artisan db:seed --force 2>&1";
    exec($command, $output, $return);
    
    sendProgress(80, 'Dados iniciais carregados', '✅ Dados iniciais carregados');
    
    // 7. Criar usuário admin
    sendProgress(85, 'Criando administrador...', '👤 Criando usuário administrador...');
    
    $command = "cd " . escapeshellarg($rootDir) . " && php artisan portal:create-admin --name=\"" . escapeshellarg($config['admin_name']) . "\" --email=\"" . escapeshellarg($config['admin_email']) . "\" --password=\"" . escapeshellarg($config['admin_password']) . "\" 2>&1";
    exec($command, $output, $return);
    
    if ($return !== 0) {
        sendProgress(0, 'Erro ao criar admin', '❌ Erro ao criar administrador: ' . implode("\n", array_slice($output, -5)));
        exit;
    }
    
    sendProgress(90, 'Admin criado', '✅ Usuário administrador criado');
    
    // 8. Criar link simbólico do storage
    sendProgress(92, 'Configurando storage...', '📁 Criando link simbólico do storage...');
    
    $command = "cd " . escapeshellarg($rootDir) . " && php artisan storage:link --force 2>&1";
    exec($command, $output, $return);
    
    sendProgress(95, 'Storage configurado', '✅ Storage configurado');
    
    // 9. Configurar permissões
    sendProgress(96, 'Configurando permissões...', '🔐 Configurando permissões...');
    
    $dirs = ['storage', 'bootstrap/cache'];
    foreach ($dirs as $dir) {
        $path = $rootDir . '/' . $dir;
        if (is_dir($path)) {
            chmod($path, 0755);
            // Recursivo
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($path, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::SELF_FIRST
            );
            foreach ($iterator as $item) {
                chmod($item->getPathname(), is_dir($item) ? 0755 : 0644);
            }
        }
    }
    
    sendProgress(98, 'Permissões configuradas', '✅ Permissões configuradas');
    
    // 10. Cache e otimização
    sendProgress(99, 'Otimizando...', '⚡ Otimizando aplicação...');
    
    $commands = [
        'config:cache',
        'route:cache',
        'view:cache'
    ];
    
    foreach ($commands as $cmd) {
        $command = "cd " . escapeshellarg($rootDir) . " && php artisan $cmd 2>&1";
        exec($command, $output, $return);
    }
    
    sendProgress(100, 'Instalação concluída!', '🎉 Instalação concluída com sucesso!');
    
} else {
    sendProgress(0, 'Ação inválida', '❌ Ação não reconhecida');
}

