<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AutoPostJob;
use App\Models\AutoPostLog;
use App\Services\AutoPostService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class AutoPostController extends Controller
{
    protected $autoPostService;

    public function __construct(AutoPostService $autoPostService)
    {
        $this->middleware('auth');
        $this->autoPostService = $autoPostService;
    }

    public function index()
    {
        $jobs = AutoPostJob::with('user')->orderBy('created_at', 'desc')->get();
        $providers = json_decode(file_get_contents(database_path('providers.json')), true);
        $adapters = $this->autoPostService->getAvailableAdapters();
        $categories = \App\Models\Category::active()->get();

        return view('admin.autopost.index', compact('jobs', 'providers', 'adapters', 'categories'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'provider_name' => 'required|string|max:255',
            'provider_type' => 'required|string|in:rss,dino,agencia_x,custom',
            'provider_url' => 'required|url',
            'provider_config' => 'nullable|array',
            'frequency' => 'required|integer|min:10',
            'max_import' => 'required|integer|min:1|max:10',
            'default_category_id' => 'required|exists:categories,id',
            'create_category_if_not_exists' => 'boolean',
            'auto_publish' => 'required|in:draft,published',
            'send_notifications' => 'boolean',
            'publish_social' => 'boolean',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['user_id'] = Auth::id();
        $validated['provider_config'] = $validated['provider_config'] ?? [];

        AutoPostJob::create($validated);

        return redirect()->route('admin.autopost.index')
            ->with('success', 'Job de AutoPost criado com sucesso!');
    }

    public function update(Request $request, AutoPostJob $job)
    {
        $validated = $request->validate([
            'provider_name' => 'required|string|max:255',
            'provider_type' => 'required|string|in:rss,dino,agencia_x,custom',
            'provider_url' => 'required|url',
            'provider_config' => 'nullable|array',
            'frequency' => 'required|integer|min:10',
            'max_import' => 'required|integer|min:1|max:10',
            'default_category_id' => 'required|exists:categories,id',
            'create_category_if_not_exists' => 'boolean',
            'auto_publish' => 'required|in:draft,published',
            'send_notifications' => 'boolean',
            'publish_social' => 'boolean',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['provider_config'] = $validated['provider_config'] ?? [];

        $job->update($validated);

        return redirect()->route('admin.autopost.index')
            ->with('success', 'Job atualizado com sucesso!');
    }

    public function destroy(AutoPostJob $job)
    {
        $job->delete();

        return redirect()->route('admin.autopost.index')
            ->with('success', 'Job excluído com sucesso!');
    }

    public function run(AutoPostJob $job)
    {
        try {
            $result = $this->autoPostService->runJob($job);
            
            return redirect()->route('admin.autopost.index')
                ->with('success', "Importação concluída! {$result['imported']} notícias importadas, {$result['skipped']} ignoradas.");
        } catch (\Exception $e) {
            return redirect()->route('admin.autopost.index')
                ->with('error', 'Erro ao executar importação: ' . $e->getMessage());
        }
    }

    public function logs(AutoPostJob $job)
    {
        $logs = AutoPostLog::where('job_id', $job->id)
            ->orderBy('created_at', 'desc')
            ->paginate(50);

        return view('admin.autopost.logs', compact('job', 'logs'));
    }
}

