<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classified;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ClassifiedController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $query = Classified::with(['user', 'category']);

        if ($request->filled('search')) {
            $query->where('title', 'like', "%{$request->search}%");
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $classifieds = $query->orderBy('created_at', 'desc')->paginate(20);
        $categories = Category::active()->get();

        return view('admin.classifieds.index', compact('classifieds', 'categories'));
    }

    public function create()
    {
        $categories = Category::active()->orderBy('name')->get();
        return view('admin.classifieds.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'description' => 'required|string',
            'price' => 'nullable|numeric|min:0',
            'contact_name' => 'required|string|max:255',
            'contact_email' => 'nullable|email',
            'contact_phone' => 'nullable|string|max:20',
            'contact_whatsapp' => 'nullable|string|max:20',
            'images' => 'nullable|array|max:10',
            'images.*' => 'image|max:2048',
            'status' => 'required|in:pending,active,expired,sold',
            'expires_at' => 'nullable|date',
        ]);

        $validated['slug'] = Str::slug($validated['title']);
        $validated['user_id'] = Auth::id();

        // Upload de imagens
        $images = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $images[] = $image->store('classifieds', 'public');
            }
        }
        $validated['images'] = $images;

        Classified::create($validated);

        return redirect()->route('admin.classifieds.index')
            ->with('success', 'Classificado criado com sucesso!');
    }

    public function edit(Classified $classified)
    {
        $categories = Category::active()->orderBy('name')->get();
        return view('admin.classifieds.edit', compact('classified', 'categories'));
    }

    public function update(Request $request, Classified $classified)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'category_id' => 'nullable|exists:categories,id',
            'description' => 'required|string',
            'price' => 'nullable|numeric|min:0',
            'contact_name' => 'required|string|max:255',
            'contact_email' => 'nullable|email',
            'contact_phone' => 'nullable|string|max:20',
            'contact_whatsapp' => 'nullable|string|max:20',
            'images' => 'nullable|array|max:10',
            'images.*' => 'image|max:2048',
            'status' => 'required|in:pending,active,expired,sold',
            'expires_at' => 'nullable|date',
        ]);

        if ($classified->title !== $validated['title']) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        // Atualizar imagens
        $images = $classified->images ?? [];
        if ($request->hasFile('images')) {
            // Deletar imagens antigas se solicitado
            foreach ($request->input('delete_images', []) as $imagePath) {
                Storage::disk('public')->delete($imagePath);
                $images = array_diff($images, [$imagePath]);
            }
            
            // Adicionar novas imagens
            foreach ($request->file('images') as $image) {
                $images[] = $image->store('classifieds', 'public');
            }
        }
        $validated['images'] = array_values($images);

        $classified->update($validated);

        return redirect()->route('admin.classifieds.index')
            ->with('success', 'Classificado atualizado com sucesso!');
    }

    public function destroy(Classified $classified)
    {
        // Deletar imagens
        if ($classified->images) {
            foreach ($classified->images as $image) {
                Storage::disk('public')->delete($image);
            }
        }

        $classified->delete();

        return redirect()->route('admin.classifieds.index')
            ->with('success', 'Classificado movido para a lixeira!');
    }

    public function approve(Classified $classified)
    {
        $classified->update([
            'status' => 'active',
            'expires_at' => now()->addDays(30), // 30 dias de validade
        ]);

        return back()->with('success', 'Classificado aprovado com sucesso!');
    }
}

