<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Models\Video;
use App\Models\Gallery;
use App\Models\Classified;
use App\Models\AdCampaign;
use App\Services\MasterPanelService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    protected $masterPanel;

    public function __construct(MasterPanelService $masterPanel)
    {
        $this->middleware('auth');
        $this->masterPanel = $masterPanel;
    }

    public function index()
    {
        // Estatísticas gerais
        $stats = [
            'news_today' => News::whereDate('created_at', today())->count(),
            'news_total' => News::count(),
            'news_published' => News::where('status', 'published')->count(),
            'videos_total' => Video::count(),
            'galleries_total' => Gallery::count(),
            'classifieds_pending' => Classified::where('status', 'pending')->count(),
            'classifieds_active' => Classified::where('status', 'active')->count(),
            'ads_active' => AdCampaign::where('status', 'active')->count(),
        ];

        // Notícias recentes
        $recentNews = News::with(['user', 'category'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Usuários recentes
        $recentUsers = DB::table('users')
            ->whereNotNull('last_login')
            ->orderBy('last_login', 'desc')
            ->limit(5)
            ->get();

        // Atividades recentes
        $recentActivities = DB::table('activity_logs')
            ->with(['user'])
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Verificar atualizações do Master
        $updates = [];
        $newPlugins = [];
        try {
            $updates = $this->masterPanel->checkUpdates(config('app.version', '1.0.0')) ?? [];
            $newPlugins = $this->masterPanel->getNewPlugins() ?? [];
        } catch (\Exception $e) {
            // Ignorar erro silenciosamente
        }

        // Google Trends
        $trends = $this->getGoogleTrends();

        return view('admin.dashboard.index', compact(
            'stats',
            'recentNews',
            'recentUsers',
            'recentActivities',
            'updates',
            'trends',
            'newPlugins'
        ));
    }

    /**
     * Obter saudação por período
     */
    public static function getGreeting()
    {
        $hour = (int) date('H');
        
        if ($hour >= 5 && $hour < 12) {
            return 'Bom dia';
        } elseif ($hour >= 12 && $hour < 18) {
            return 'Boa tarde';
        } else {
            return 'Boa noite';
        }
    }

    protected function getGoogleTrends()
    {
        try {
            // Tentar usar serviço de Google Trends se existir
            if (class_exists(\App\Services\GoogleTrendsService::class)) {
                $trendsService = app(\App\Services\GoogleTrendsService::class);
                return $trendsService->getTrendingTopics('BR', 7);
            }
        } catch (\Exception $e) {
            // Fallback para dados estáticos
        }

        // Fallback: retornar alguns tópicos relevantes do Brasil
        return [
            ['title' => 'Política Brasileira', 'traffic' => 'Alto'],
            ['title' => 'Economia do Brasil', 'traffic' => 'Médio'],
            ['title' => 'Esportes Nacionais', 'traffic' => 'Alto'],
            ['title' => 'Tecnologia no Brasil', 'traffic' => 'Médio'],
            ['title' => 'Cultura Brasileira', 'traffic' => 'Médio'],
            ['title' => 'Saúde Pública', 'traffic' => 'Alto'],
            ['title' => 'Educação no Brasil', 'traffic' => 'Médio'],
        ];
    }
}

