<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Gallery;
use App\Models\GalleryImage;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class GalleryController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $query = Gallery::with(['user', 'images']);

        if ($request->filled('search')) {
            $query->where('title', 'like', "%{$request->search}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $galleries = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.galleries.index', compact('galleries'));
    }

    public function create()
    {
        return view('admin.galleries.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'featured_image' => 'nullable|image|max:2048',
            'images' => 'required|array|min:1',
            'images.*' => 'image|max:2048',
            'status' => 'required|in:active,inactive',
            'published_at' => 'nullable|date',
        ]);

        $validated['slug'] = Str::slug($validated['title']);
        $validated['user_id'] = Auth::id();

        // Upload da imagem destacada
        if ($request->hasFile('featured_image')) {
            $validated['featured_image'] = $request->file('featured_image')->store('galleries', 'public');
        }

        $gallery = Gallery::create($validated);

        // Upload das imagens
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $index => $image) {
                $path = $image->store('galleries/images', 'public');
                GalleryImage::create([
                    'gallery_id' => $gallery->id,
                    'image' => $path,
                    'order' => $index,
                ]);
            }
        }

        return redirect()->route('admin.galleries.index')
            ->with('success', 'Galeria criada com sucesso!');
    }

    public function edit(Gallery $gallery)
    {
        $gallery->load('images');
        return view('admin.galleries.edit', compact('gallery'));
    }

    public function update(Request $request, Gallery $gallery)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'featured_image' => 'nullable|image|max:2048',
            'images' => 'nullable|array',
            'images.*' => 'image|max:2048',
            'status' => 'required|in:active,inactive',
            'published_at' => 'nullable|date',
        ]);

        if ($gallery->title !== $validated['title']) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        if ($request->hasFile('featured_image')) {
            if ($gallery->featured_image) {
                Storage::disk('public')->delete($gallery->featured_image);
            }
            $validated['featured_image'] = $request->file('featured_image')->store('galleries', 'public');
        }

        $gallery->update($validated);

        // Adicionar novas imagens
        if ($request->hasFile('images')) {
            $maxOrder = $gallery->images()->max('order') ?? -1;
            foreach ($request->file('images') as $image) {
                $path = $image->store('galleries/images', 'public');
                GalleryImage::create([
                    'gallery_id' => $gallery->id,
                    'image' => $path,
                    'order' => ++$maxOrder,
                ]);
            }
        }

        return redirect()->route('admin.galleries.index')
            ->with('success', 'Galeria atualizada com sucesso!');
    }

    public function destroy(Gallery $gallery)
    {
        // Deletar imagens
        foreach ($gallery->images as $image) {
            Storage::disk('public')->delete($image->image);
        }
        if ($gallery->featured_image) {
            Storage::disk('public')->delete($gallery->featured_image);
        }

        $gallery->delete();

        return redirect()->route('admin.galleries.index')
            ->with('success', 'Galeria movida para a lixeira!');
    }

    public function deleteImage(GalleryImage $image)
    {
        Storage::disk('public')->delete($image->image);
        $image->delete();

        return back()->with('success', 'Imagem excluída com sucesso!');
    }
}

