<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Models\Category;
use App\Models\Tag;
use App\Models\Municipality;
use App\Models\Gallery;
use App\Models\NewsRevision;
use App\Services\WhatsAppService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class NewsController extends Controller
{
    protected $whatsapp;

    public function __construct(WhatsAppService $whatsapp)
    {
        $this->middleware('auth');
        $this->whatsapp = $whatsapp;
    }

    /**
     * Listar notícias
     */
    public function index(Request $request)
    {
        $query = News::with(['user', 'category']);

        // Filtros
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('body', 'like', "%{$search}%");
            });
        }

        if ($request->filled('category_id')) {
            $query->where('category_id', $request->category_id);
        }

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        if ($request->filled('is_featured')) {
            $query->where('is_featured', $request->is_featured);
        }

        if ($request->filled('municipality_id')) {
            $query->where('municipality_id', $request->municipality_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('published_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('published_at', '<=', $request->date_to);
        }

        if ($request->filled('column')) {
            $query->where('column', 'like', "%{$request->column}%");
        }

        $news = $query->orderBy('created_at', 'desc')->paginate(20);
        $categories = Category::all();
        $users = \App\Models\User::all();
        $municipalities = Municipality::all();

        return view('admin.news.index', compact('news', 'categories', 'users', 'municipalities'));

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('is_featured')) {
            $query->where('is_featured', $request->is_featured);
        }

        if ($request->filled('municipality_id')) {
            $query->where('municipality_id', $request->municipality_id);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        $news = $query->orderBy('created_at', 'desc')->paginate(20);
        $categories = Category::active()->get();
        $users = \App\Models\User::active()->get();
        $municipalities = \App\Models\Municipality::active()->get();

        return view('admin.news.index', compact('news', 'categories', 'users', 'municipalities'));
    }

    /**
     * Formulário de criação
     */
    public function create()
    {
        $categories = Category::active()->orderBy('name')->get();
        $tags = Tag::orderBy('name')->get();
        $municipalities = Municipality::active()->orderBy('name')->get();
        $galleries = Gallery::active()->orderBy('title')->get();

        return view('admin.news.create', compact('categories', 'tags', 'municipalities', 'galleries'));
    }

    /**
     * Salvar notícia
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'body' => 'required|string',
            'cover_image' => 'nullable|image|max:2048',
            'status' => 'required|in:draft,scheduled,published,pending_review',
            'published_at' => 'nullable|date',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
        ]);

        // Gerar slug
        $validated['slug'] = Str::slug($validated['title']);
        $validated['user_id'] = Auth::id();

        // Upload da imagem de capa
        if ($request->hasFile('cover_image')) {
            $validated['cover_image'] = $request->file('cover_image')->store('news', 'public');
        }

        // Criar excerpt se não fornecido
        if (empty($validated['excerpt'])) {
            $validated['excerpt'] = Str::limit(strip_tags($validated['body']), 200);
        }

        // Campos booleanos
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_slide'] = $request->has('is_slide');
        $validated['is_headline'] = $request->has('is_headline');
        $validated['is_more_news'] = $request->has('is_more_news');
        $validated['is_urgent'] = $request->has('is_urgent');

        // Criar notícia
        $news = News::create($validated);

        // Adicionar tags
        if ($request->filled('tags')) {
            $news->tags()->attach($request->tags);
        }

        // Criar revisão
        $this->createRevision($news);

        // Notificações se publicado
        if ($validated['status'] === 'published') {
            $this->handleNotifications($news, $request);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'Notícia criada com sucesso!');
    }

    /**
     * Formulário de edição
     */
    public function edit(News $news)
    {
        $categories = Category::active()->orderBy('name')->get();
        $tags = Tag::orderBy('name')->get();
        $municipalities = Municipality::active()->orderBy('name')->get();
        $galleries = Gallery::active()->orderBy('title')->get();
        $revisions = $news->revisions()->with('user')->orderBy('created_at', 'desc')->get();

        return view('admin.news.edit', compact('news', 'categories', 'tags', 'municipalities', 'galleries', 'revisions'));
    }

    /**
     * Atualizar notícia
     */
    public function update(Request $request, News $news)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'category_id' => 'required|exists:categories,id',
            'body' => 'required|string',
            'cover_image' => 'nullable|image|max:2048',
            'status' => 'required|in:draft,scheduled,published,pending_review',
            'published_at' => 'nullable|date',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
        ]);

        // Atualizar slug se título mudou
        if ($news->title !== $validated['title']) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        // Upload da nova imagem
        if ($request->hasFile('cover_image')) {
            // Deletar imagem antiga
            if ($news->cover_image) {
                Storage::disk('public')->delete($news->cover_image);
            }
            $validated['cover_image'] = $request->file('cover_image')->store('news', 'public');
        }

        // Campos booleanos
        $validated['is_featured'] = $request->has('is_featured');
        $validated['is_slide'] = $request->has('is_slide');
        $validated['is_headline'] = $request->has('is_headline');
        $validated['is_more_news'] = $request->has('is_more_news');
        $validated['is_urgent'] = $request->has('is_urgent');

        // Atualizar
        $news->update($validated);

        // Atualizar tags
        if ($request->filled('tags')) {
            $news->tags()->sync($request->tags);
        } else {
            $news->tags()->detach();
        }

        // Criar nova revisão se houver mudanças significativas
        if ($news->wasChanged(['title', 'body'])) {
            $this->createRevision($news);
        }

        // Notificações se publicado agora
        if ($validated['status'] === 'published' && $news->wasChanged('status')) {
            $this->handleNotifications($news, $request);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'Notícia atualizada com sucesso!');
    }

    /**
     * Duplicar notícia
     */
    public function duplicate(News $news)
    {
        $newNews = $news->replicate();
        $newNews->title = $news->title . ' (Cópia)';
        $newNews->slug = $news->slug . '-' . time();
        $newNews->status = 'draft';
        $newNews->views = 0;
        $newNews->user_id = Auth::id();
        $newNews->save();

        // Duplicar tags
        $newNews->tags()->attach($news->tags->pluck('id'));

        return redirect()->route('admin.news.edit', $newNews)
            ->with('success', 'Notícia duplicada com sucesso!');
    }

    /**
     * Deletar notícia (soft delete)
     */
    public function destroy(News $news)
    {
        $news->delete();

        return redirect()->route('admin.news.index')
            ->with('success', 'Notícia movida para a lixeira!');
    }

    /**
     * Restaurar da lixeira
     */
    public function restore($id)
    {
        $news = News::withTrashed()->findOrFail($id);
        $news->restore();

        return redirect()->route('admin.news.index')
            ->with('success', 'Notícia restaurada com sucesso!');
    }

    /**
     * Criar revisão
     */
    protected function createRevision(News $news)
    {
        NewsRevision::create([
            'news_id' => $news->id,
            'user_id' => Auth::id(),
            'title' => $news->title,
            'body' => $news->body,
        ]);
    }

    /**
     * Lidar com notificações
     */
    protected function handleNotifications(News $news, Request $request)
    {
        // Notificação Push
        if ($request->has('notification_push')) {
            // Implementar notificação push
        }

        // Notificação WhatsApp
        if ($request->has('notification_whatsapp')) {
            // Implementar notificação WhatsApp
        }

        // Publicar no Instagram
        if ($request->has('publish_instagram')) {
            // Implementar publicação no Instagram
        }

        // Publicar no Facebook
        if ($request->has('publish_facebook')) {
            // Implementar publicação no Facebook
        }
    }

    /**
     * Duplicar notícia
     */
    public function duplicate(News $news)
    {
        $newNews = $news->replicate();
        $newNews->title = $news->title . ' (Cópia)';
        $newNews->slug = Str::slug($newNews->title) . '-' . time();
        $newNews->status = 'draft';
        $newNews->views = 0;
        $newNews->user_id = Auth::id();
        $newNews->save();

        // Duplicar tags
        if ($news->tags()->count() > 0) {
            $newNews->tags()->attach($news->tags->pluck('id'));
        }

        if (request()->expectsJson()) {
            return response()->json(['success' => true, 'message' => 'Notícia duplicada com sucesso!']);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'Notícia duplicada com sucesso!');
    }

    /**
     * Alterar status
     */
    public function changeStatus(Request $request, News $news)
    {
        $validated = $request->validate([
            'status' => 'required|in:draft,scheduled,published,pending_review',
        ]);

        $news->update(['status' => $validated['status']]);

        if ($request->expectsJson()) {
            return response()->json(['success' => true, 'message' => 'Status alterado com sucesso!']);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'Status alterado com sucesso!');
    }

    /**
     * Ação em massa
     */
    public function bulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:publish,draft,delete,trash',
            'ids' => 'required|array',
            'ids.*' => 'exists:news,id',
        ]);

        $news = News::whereIn('id', $validated['ids']);

        switch ($validated['action']) {
            case 'publish':
                $news->update(['status' => 'published']);
                $message = 'Notícias publicadas com sucesso!';
                break;
            case 'draft':
                $news->update(['status' => 'draft']);
                $message = 'Notícias movidas para rascunho!';
                break;
            case 'delete':
                $news->delete();
                $message = 'Notícias excluídas permanentemente!';
                break;
            case 'trash':
                $news->delete();
                $message = 'Notícias movidas para a lixeira!';
                break;
        }

        if ($request->expectsJson()) {
            return response()->json(['success' => true, 'message' => $message]);
        }

        return redirect()->route('admin.news.index')
            ->with('success', $message);
    }

    /**
     * Mover para lixeira
     */
    public function trash(News $news)
    {
        $news->delete();

        if (request()->expectsJson()) {
            return response()->json(['success' => true, 'message' => 'Notícia movida para a lixeira!']);
        }

        return redirect()->route('admin.news.index')
            ->with('success', 'Notícia movida para a lixeira!');
    }
}

