<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\InstalledPlugin;
use App\Services\MasterPanelService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use ZipArchive;

class PluginController extends Controller
{
    protected $masterPanel;

    public function __construct(MasterPanelService $masterPanel)
    {
        $this->middleware('auth');
        $this->masterPanel = $masterPanel;
    }

    public function index()
    {
        // Buscar plugins disponíveis do Master Panel
        $availablePlugins = $this->masterPanel->getAvailablePlugins();
        $plugins = $availablePlugins['plugins'] ?? [];

        // Buscar plugins instalados
        $installed = InstalledPlugin::all()->keyBy('plugin_slug');

        return view('admin.plugins.index', compact('plugins', 'installed'));
    }

    public function install(Request $request, $pluginSlug)
    {
        $request->validate([
            'license_key' => 'nullable|string',
        ]);

        // Verificar se já está instalado
        $installed = InstalledPlugin::where('plugin_slug', $pluginSlug)->first();
        if ($installed) {
            return back()->with('error', 'Plugin já está instalado!');
        }

        // Ativar plugin no Master Panel
        $result = $this->masterPanel->activatePlugin($pluginSlug, $request->license_key);

        if (!$result || !isset($result['success']) || !$result['success']) {
            return back()->with('error', $result['error'] ?? 'Erro ao instalar plugin');
        }

        // Buscar informações do plugin
        $availablePlugins = $this->masterPanel->getAvailablePlugins();
        $pluginInfo = collect($availablePlugins['plugins'] ?? [])
            ->firstWhere('slug', $pluginSlug);

        if (!$pluginInfo) {
            return back()->with('error', 'Plugin não encontrado');
        }

        // Salvar plugin instalado
        InstalledPlugin::create([
            'plugin_slug' => $pluginSlug,
            'plugin_name' => $pluginInfo['name'],
            'version' => $pluginInfo['version'],
            'license_key' => $request->license_key,
            'status' => 'active',
            'installed_at' => now(),
        ]);

        return back()->with('success', 'Plugin instalado com sucesso!');
    }

    public function uninstall(InstalledPlugin $plugin)
    {
        $plugin->delete();

        return back()->with('success', 'Plugin desinstalado com sucesso!');
    }

    public function toggle(InstalledPlugin $plugin)
    {
        $plugin->update([
            'status' => $plugin->status === 'active' ? 'inactive' : 'active'
        ]);

        return back()->with('success', 'Status do plugin alterado!');
    }

    public function configure(InstalledPlugin $plugin)
    {
        return view('admin.plugins.configure', compact('plugin'));
    }

    public function updateConfig(Request $request, InstalledPlugin $plugin)
    {
        $config = $plugin->config ?? [];
        $newConfig = array_merge($config, $request->except(['_token', '_method']));

        $plugin->update(['config' => $newConfig]);

        return back()->with('success', 'Configurações salvas com sucesso!');
    }

    public function upload(Request $request)
    {
        $request->validate([
            'plugin_file' => 'required|file|mimes:zip|max:10240',
        ]);

        $file = $request->file('plugin_file');
        $zipPath = $file->store('plugins', 'local');

        // Extrair e instalar plugin
        $zip = new ZipArchive();
        if ($zip->open(storage_path('app/' . $zipPath)) === TRUE) {
            $extractPath = storage_path('app/plugins/' . pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME));
            $zip->extractTo($extractPath);
            $zip->close();

            // Ler informações do plugin (plugin.json)
            $pluginJsonPath = $extractPath . '/plugin.json';
            if (file_exists($pluginJsonPath)) {
                $pluginInfo = json_decode(file_get_contents($pluginJsonPath), true);
                
                InstalledPlugin::create([
                    'plugin_slug' => $pluginInfo['slug'] ?? pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME),
                    'plugin_name' => $pluginInfo['name'] ?? 'Plugin Manual',
                    'version' => $pluginInfo['version'] ?? '1.0.0',
                    'status' => 'active',
                    'installed_at' => now(),
                ]);

                return redirect()->route('admin.plugins.index')
                    ->with('success', 'Plugin instalado com sucesso!');
            }
        }

        return back()->with('error', 'Erro ao processar arquivo ZIP do plugin');
    }
}

