<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Poll;
use App\Models\PollOption;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PollController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $query = Poll::with(['user', 'options']);

        if ($request->filled('search')) {
            $query->where('title', 'like', "%{$request->search}%");
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $polls = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.polls.index', compact('polls'));
    }

    public function create()
    {
        return view('admin.polls.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'status' => 'required|in:active,draft,closed',
            'expires_at' => 'nullable|date',
            'options' => 'required|array|min:2',
            'options.*' => 'required|string|max:255',
        ]);

        $validated['slug'] = Str::slug($validated['title']);
        $validated['user_id'] = Auth::id();

        if ($request->hasFile('image')) {
            $validated['image'] = $request->file('image')->store('polls', 'public');
        }

        $poll = Poll::create($validated);

        // Criar opções
        foreach ($request->options as $index => $option) {
            PollOption::create([
                'poll_id' => $poll->id,
                'option' => $option,
                'order' => $index,
            ]);
        }

        return redirect()->route('admin.polls.index')
            ->with('success', 'Enquete criada com sucesso!');
    }

    public function edit(Poll $poll)
    {
        $poll->load('options');
        return view('admin.polls.edit', compact('poll'));
    }

    public function update(Request $request, Poll $poll)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'image' => 'nullable|image|max:2048',
            'status' => 'required|in:active,draft,closed',
            'expires_at' => 'nullable|date',
            'options' => 'nullable|array|min:2',
            'options.*' => 'nullable|string|max:255',
        ]);

        if ($poll->title !== $validated['title']) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        if ($request->hasFile('image')) {
            if ($poll->image) {
                Storage::disk('public')->delete($poll->image);
            }
            $validated['image'] = $request->file('image')->store('polls', 'public');
        }

        $poll->update($validated);

        // Atualizar opções se fornecido
        if ($request->filled('options')) {
            // Deletar opções antigas (se não tiver votos)
            $poll->options()->where('votes', 0)->delete();
            
            // Adicionar novas opções
            $maxOrder = $poll->options()->max('order') ?? -1;
            foreach ($request->options as $option) {
                if (!empty($option)) {
                    PollOption::create([
                        'poll_id' => $poll->id,
                        'option' => $option,
                        'order' => ++$maxOrder,
                    ]);
                }
            }
        }

        return redirect()->route('admin.polls.index')
            ->with('success', 'Enquete atualizada com sucesso!');
    }

    public function destroy(Poll $poll)
    {
        $poll->options()->delete();
        if ($poll->image) {
            Storage::disk('public')->delete($poll->image);
        }
        $poll->delete();

        return redirect()->route('admin.polls.index')
            ->with('success', 'Enquete excluída com sucesso!');
    }

    public function results(Poll $poll)
    {
        $poll->load('options');
        return view('admin.polls.results', compact('poll'));
    }

    public function exportCsv(Poll $poll)
    {
        $poll->load('options');
        
        $filename = 'enquete_' . $poll->slug . '_' . date('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"{$filename}\"",
        ];

        $callback = function() use ($poll) {
            $file = fopen('php://output', 'w');
            
            // Cabeçalho
            fputcsv($file, ['Opção', 'Votos', 'Percentual']);
            
            // Dados
            foreach ($poll->options as $option) {
                fputcsv($file, [
                    $option->option,
                    $option->votes,
                    $option->percentage . '%'
                ]);
            }
            
            // Total
            fputcsv($file, ['Total', $poll->total_votes, '100%']);
            
            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}

