<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SettingsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->isAdmin()) {
                abort(403, 'Acesso negado');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $groups = [
            'general' => 'Geral',
            'seo' => 'SEO',
            'social' => 'Redes Sociais',
            'ads' => 'Anúncios',
            'email' => 'Email',
            'whatsapp' => 'WhatsApp',
            'api' => 'API Keys',
        ];

        $settings = Setting::orderBy('group')->orderBy('key')->get()->groupBy('group');

        return view('admin.settings.index', compact('settings', 'groups'));
    }

    public function update(Request $request)
    {
        $data = $request->except(['_token', '_method']);

        foreach ($data as $key => $value) {
            Setting::set($key, $value, 'string', $request->input("group_{$key}", 'general'));
        }

        // Salvar ADS.txt se fornecido
        if ($request->has('ads_txt_content')) {
            Storage::disk('public')->put('ads.txt', $request->ads_txt_content);
        }

        // Salvar scripts personalizados
        if ($request->has('head_scripts')) {
            Setting::set('head_scripts', $request->head_scripts, 'text', 'general');
        }

        if ($request->has('body_scripts')) {
            Setting::set('body_scripts', $request->body_scripts, 'text', 'general');
        }

        return back()->with('success', 'Configurações salvas com sucesso!');
    }

    public function general()
    {
        $settings = Setting::where('group', 'general')->get()->keyBy('key');
        return view('admin.settings.general', compact('settings'));
    }

    public function seo()
    {
        $settings = Setting::where('group', 'seo')->get()->keyBy('key');
        $adsTxt = Storage::disk('public')->exists('ads.txt') 
            ? Storage::disk('public')->get('ads.txt') 
            : '';

        return view('admin.settings.seo', compact('settings', 'adsTxt'));
    }

    public function scripts()
    {
        $headScripts = Setting::get('head_scripts', '');
        $bodyScripts = Setting::get('body_scripts', '');

        return view('admin.settings.scripts', compact('headScripts', 'bodyScripts'));
    }

    public function templates()
    {
        return view('admin.settings.templates');
    }

    public function updateTemplates(Request $request)
    {
        $request->validate([
            'active_template' => 'required|in:v1,v2',
        ]);

        Setting::set('active_template', $request->active_template, 'string', 'general');

        return back()->with('success', 'Template atualizado com sucesso!');
    }
}

