<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Video;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class VideoController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $query = Video::with(['user', 'category']);

        if ($request->filled('search')) {
            $query->where('title', 'like', "%{$request->search}%");
        }

        if ($request->filled('provider')) {
            $query->where('provider', $request->provider);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $videos = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('admin.videos.index', compact('videos'));
    }

    public function create()
    {
        $categories = Category::active()->orderBy('name')->get();
        return view('admin.videos.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'provider' => 'required|in:youtube,vimeo,drive,upload',
            'url' => 'required|string|max:500',
            'description' => 'nullable|string',
            'category_id' => 'nullable|exists:categories,id',
            'thumbnail' => 'nullable|image|max:1024',
            'status' => 'required|in:active,inactive',
        ]);

        $validated['slug'] = Str::slug($validated['title']);
        $validated['user_id'] = Auth::id();

        // Upload de thumbnail
        if ($request->hasFile('thumbnail')) {
            $validated['thumbnail'] = $request->file('thumbnail')->store('videos', 'public');
        }

        // Extrair embed code se necessário
        if ($validated['provider'] === 'youtube') {
            $validated['embed_code'] = $this->extractYoutubeEmbed($validated['url']);
        }

        Video::create($validated);

        return redirect()->route('admin.videos.index')
            ->with('success', 'Vídeo criado com sucesso!');
    }

    public function edit(Video $video)
    {
        $categories = Category::active()->orderBy('name')->get();
        return view('admin.videos.edit', compact('video', 'categories'));
    }

    public function update(Request $request, Video $video)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'provider' => 'required|in:youtube,vimeo,drive,upload',
            'url' => 'required|string|max:500',
            'description' => 'nullable|string',
            'category_id' => 'nullable|exists:categories,id',
            'thumbnail' => 'nullable|image|max:1024',
            'status' => 'required|in:active,inactive',
        ]);

        if ($video->title !== $validated['title']) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        if ($request->hasFile('thumbnail')) {
            if ($video->thumbnail) {
                Storage::disk('public')->delete($video->thumbnail);
            }
            $validated['thumbnail'] = $request->file('thumbnail')->store('videos', 'public');
        }

        $video->update($validated);

        return redirect()->route('admin.videos.index')
            ->with('success', 'Vídeo atualizado com sucesso!');
    }

    public function destroy(Video $video)
    {
        $video->delete();
        return redirect()->route('admin.videos.index')
            ->with('success', 'Vídeo movido para a lixeira!');
    }

    protected function extractYoutubeEmbed($url)
    {
        preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/|youtube\.com\/embed\/)([a-zA-Z0-9_-]+)/', $url, $matches);
        if (isset($matches[1])) {
            return 'https://www.youtube.com/embed/' . $matches[1];
        }
        return $url;
    }
}

