<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use App\Models\News;
use App\Models\Category;
use App\Models\Video;
use App\Models\Gallery;
use App\Models\AdCampaign;
use App\Models\AdPosition;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    public function index()
    {
        // Notícias em destaque
        $featuredNews = News::published()
            ->featured()
            ->orderBy('published_at', 'desc')
            ->limit(1)
            ->first();

        // Notícias do slide
        $slideNews = News::published()
            ->slide()
            ->orderBy('published_at', 'desc')
            ->limit(5)
            ->get();

        // Notícias principais
        $mainNews = News::published()
            ->where('is_headline', true)
            ->orderBy('published_at', 'desc')
            ->limit(4)
            ->get();

        // Notícias por categoria
        $categories = Category::active()->get();
        $newsByCategory = [];
        foreach ($categories as $category) {
            $newsByCategory[$category->id] = News::published()
                ->where('category_id', $category->id)
                ->orderBy('published_at', 'desc')
                ->limit(6)
                ->get();
        }

        // Vídeos recentes
        $recentVideos = Video::active()
            ->orderBy('created_at', 'desc')
            ->limit(6)
            ->get();

        // Galerias recentes
        $recentGalleries = Gallery::active()
            ->orderBy('published_at', 'desc')
            ->limit(3)
            ->get();

        // Anúncios
        $ads = $this->getAds();

        return view('frontend.v1.home', compact(
            'featuredNews',
            'slideNews',
            'mainNews',
            'categories',
            'newsByCategory',
            'recentVideos',
            'recentGalleries',
            'ads'
        ));
    }

    public function news($slug)
    {
        $news = News::where('slug', $slug)->published()->firstOrFail();
        
        // Incrementar visualizações
        $news->incrementViews();

        // Notícias relacionadas
        $relatedNews = News::published()
            ->where('category_id', $news->category_id)
            ->where('id', '!=', $news->id)
            ->orderBy('published_at', 'desc')
            ->limit(6)
            ->get();

        // Mais lidas
        $mostRead = News::published()
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();

        // Anúncios
        $ads = $this->getAds();

        return view('frontend.v1.news', compact('news', 'relatedNews', 'mostRead', 'ads'));
    }

    public function category($slug)
    {
        $category = Category::where('slug', $slug)->active()->firstOrFail();
        
        $news = News::published()
            ->where('category_id', $category->id)
            ->orderBy('published_at', 'desc')
            ->paginate(12);

        return view('frontend.v1.category', compact('category', 'news'));
    }

    protected function getAds()
    {
        $ads = [];
        $positions = AdPosition::active()->get();
        
        foreach ($positions as $position) {
            $campaign = AdCampaign::active()
                ->where('position_id', $position->id)
                ->inRandomOrder()
                ->first();
            
            if ($campaign) {
                $campaign->incrementImpressions();
                $ads[$position->slug] = $campaign;
            }
        }

        return $ads;
    }
}

