<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\MasterPanelService;

class InstallerController extends Controller
{
    protected $masterPanel;

    public function __construct(MasterPanelService $masterPanel)
    {
        $this->masterPanel = $masterPanel;
    }

    /**
     * Página inicial do instalador
     */
    public function index()
    {
        // Verificar se já está instalado
        if (file_exists(storage_path('installed'))) {
            return redirect('/');
        }

        return view('installer.index');
    }

    /**
     * Verificar requisitos
     */
    public function checkRequirements()
    {
        $requirements = [
            'php_version' => version_compare(PHP_VERSION, '8.1.0', '>='),
            'extensions' => $this->checkExtensions(),
            'database' => true, // Será verificado no próximo passo
            'license' => false, // Será verificado quando fornecido
        ];

        return response()->json([
            'success' => true,
            'requirements' => $requirements
        ]);
    }

    /**
     * Verificar extensões PHP
     */
    protected function checkExtensions()
    {
        $required = ['pdo', 'pdo_mysql', 'mbstring', 'xml', 'curl', 'json', 'openssl', 'zip'];
        $missing = [];

        foreach ($required as $ext) {
            if (!extension_loaded($ext)) {
                $missing[] = $ext;
            }
        }

        return empty($missing);
    }

    /**
     * Validar banco de dados
     */
    public function validateDatabase(Request $request)
    {
        $request->validate([
            'host' => 'required',
            'database' => 'required',
            'username' => 'required',
            'password' => 'nullable',
        ]);

        try {
            config([
                'database.connections.mysql.host' => $request->host,
                'database.connections.mysql.database' => $request->database,
                'database.connections.mysql.username' => $request->username,
                'database.connections.mysql.password' => $request->password,
            ]);

            DB::connection('mysql')->getPdo();

            return response()->json([
                'success' => true,
                'message' => 'Conexão com banco de dados estabelecida com sucesso!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao conectar: ' . $e->getMessage()
            ], 400);
        }
    }

    /**
     * Validar licença
     */
    public function validateLicense(Request $request)
    {
        $request->validate([
            'master_panel_url' => 'required|url',
            'license_key' => 'required|string',
        ]);

        $masterUrl = trim($request->master_panel_url);
        
        // Corrigir URL se necessário
        if (!preg_match('/^https?:\/\//', $masterUrl)) {
            $masterUrl = 'https://' . $masterUrl;
        }
        
        // Garantir que URLs do canal33 terminem com .com.br
        if (strpos($masterUrl, 'canal33.com') !== false) {
            // Corrigir .com.b ou .com para .com.br
            $masterUrl = preg_replace('/\.com\.?b?r?(\/|$)/', '.com.br$1', $masterUrl);
        }
        
        $masterUrl = rtrim($masterUrl, '/');
        $licenseKey = trim($request->license_key);

        // Validar formato da URL
        if (!filter_var($masterUrl, FILTER_VALIDATE_URL)) {
            return response()->json([
                'success' => false,
                'message' => 'URL do Master Panel inválida. Verifique se está completa (ex: https://master.canal33.com.br)'
            ], 400);
        }
        
        // Validar que a URL termina com domínio válido
        if (!preg_match('/\.(com|net|org|br)(\.br)?(\/|$)/', $masterUrl)) {
            return response()->json([
                'success' => false,
                'message' => 'URL do Master Panel parece estar incompleta. Verifique se termina com .com.br (ex: https://master.canal33.com.br)'
            ], 400);
        }

        // Validar formato da license key
        if (!preg_match('/^PVX-[A-F0-9]{8}-[A-F0-9]{8}-[A-F0-9]{8}$/i', $licenseKey)) {
            return response()->json([
                'success' => false,
                'message' => 'Formato de License Key inválido. Deve ser no formato: PVX-XXXXXXXX-XXXXXXXX-XXXXXXXX'
            ], 400);
        }

        try {
            $url = $masterUrl . '/api/v1/validate-license';
            $params = [
                'license_key' => $licenseKey,
                'ip' => $request->ip(),
                'domain' => $request->getHost()
            ];

            Log::info('Validating license during installation', [
                'url' => $url,
                'license_key' => substr($licenseKey, 0, 10) . '...'
            ]);

            $response = Http::timeout(15)
                ->retry(3, 500)
                ->acceptJson()
                ->get($url, $params);

            Log::info('License validation response', [
                'status' => $response->status(),
                'body' => $response->json()
            ]);

            if ($response->successful()) {
                $result = $response->json();

                if (isset($result['success']) && $result['success']) {
                    // Salvar configurações temporariamente
                    session([
                        'install.master_url' => $masterUrl,
                        'install.license_key' => $licenseKey,
                        'install.license_data' => $result['license'] ?? null
                    ]);

                    return response()->json([
                        'success' => true,
                        'message' => 'Licença validada com sucesso!',
                        'license' => $result['license'] ?? null
                    ]);
                } else {
                    return response()->json([
                        'success' => false,
                        'message' => $result['message'] ?? 'Licença inválida ou expirada'
                    ], 403);
                }
            } else {
                $status = $response->status();
                $body = $response->json();

                $errorMessage = $body['message'] ?? "Erro ao validar licença (Status: {$status})";

                if ($status === 404) {
                    $errorMessage .= '. Endpoint não encontrado. Verifique se a URL do Master Panel está correta e se o endpoint /api/v1/validate-license existe.';
                } elseif ($status === 403) {
                    $errorMessage .= '. Licença inválida, expirada ou bloqueada.';
                } elseif ($status >= 500) {
                    $errorMessage .= '. Erro no servidor Master Panel. Tente novamente mais tarde.';
                } else {
                    $errorMessage .= '. Verifique a URL do Master Panel e sua conexão com a internet.';
                }

                return response()->json([
                    'success' => false,
                    'message' => $errorMessage,
                    'debug' => [
                        'url' => $url,
                        'status' => $status,
                        'response' => $body
                    ]
                ], $status);
            }
        } catch (\Illuminate\Http\Client\ConnectionException $e) {
            Log::error('License validation connection error', [
                'url' => $url ?? 'N/A',
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro de conexão com o Master Panel. Verifique se a URL está correta (' . $masterUrl . ') e se há conexão com a internet. Erro: ' . $e->getMessage()
            ], 503);
        } catch (\Exception $e) {
            Log::error('License validation error', [
                'url' => $url ?? 'N/A',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Erro ao validar licença: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Criar usuário admin
     */
    public function createAdmin(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        try {
            // Criar usuário admin
            $userId = DB::table('users')->insertGetId([
                'name' => $request->name,
                'email' => $request->email,
                'password' => bcrypt($request->password),
                'role' => 'admin',
                'email_verified_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Usuário admin criado com sucesso!',
                'user_id' => $userId
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao criar usuário: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Finalizar instalação
     */
    public function finish(Request $request)
    {
        try {
            // Salvar configurações no .env
            $envPath = base_path('.env');
            $envContent = File::get($envPath);

            // Atualizar MASTER_PANEL_URL
            $masterUrl = session('install.master_url', 'https://master.canal33.com.br');
            $envContent = preg_replace(
                '/MASTER_PANEL_URL=.*/',
                'MASTER_PANEL_URL=' . $masterUrl,
                $envContent
            );

            // Atualizar MASTER_PANEL_LICENSE_KEY
            $licenseKey = session('install.license_key');
            if ($licenseKey) {
                if (strpos($envContent, 'MASTER_PANEL_LICENSE_KEY=') === false) {
                    $envContent .= "\nMASTER_PANEL_LICENSE_KEY={$licenseKey}\n";
                } else {
                    $envContent = preg_replace(
                        '/MASTER_PANEL_LICENSE_KEY=.*/',
                        'MASTER_PANEL_LICENSE_KEY=' . $licenseKey,
                        $envContent
                    );
                }
            }

            File::put($envPath, $envContent);

            // Criar arquivo de instalação completa
            File::put(storage_path('installed'), date('Y-m-d H:i:s'));

            // Limpar sessão
            session()->forget('install');

            return response()->json([
                'success' => true,
                'message' => 'Instalação concluída com sucesso!'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Erro ao finalizar instalação: ' . $e->getMessage()
            ], 500);
        }
    }
}

