<?php

namespace App\Http\Controllers;

use App\Models\News;
use App\Models\Category;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class NewsController extends Controller
{
    /**
     * Exibir notícia - Template V1
     */
    public function showV1($slug)
    {
        $news = News::where('slug', $slug)
            ->where('status', 'published')
            ->with(['category', 'user', 'tags', 'gallery'])
            ->firstOrFail();

        // Incrementar visualizações
        $news->increment('views');

        // Notícias relacionadas (mesma categoria)
        $relatedNews = News::where('status', 'published')
            ->where('category_id', $news->category_id)
            ->where('id', '!=', $news->id)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->limit(6)
            ->get();

        // Mais lidas
        $mostRead = News::where('status', 'published')
            ->where('id', '!=', $news->id)
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();

        // Mais recentes
        $latestNews = News::where('status', 'published')
            ->where('id', '!=', $news->id)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->limit(5)
            ->get();

        // Vídeos recentes para sidebar
        $recentVideos = \App\Models\Video::where('status', 'published')
            ->latest('published_at')
            ->limit(3)
            ->get();

        // Anúncios
        $sidebarAds = $this->getSidebarAds();

        // Passar latestNews como latestPosts para compatibilidade
        $latestPosts = $latestNews;
        $ads = $sidebarAds;

        return view('frontend.v1.news', compact(
            'news',
            'relatedNews',
            'mostRead',
            'latestPosts',
            'recentVideos',
            'ads'
        ));
    }

    /**
     * Exibir notícia - Template V2
     */
    public function showV2($slug)
    {
        $news = News::where('slug', $slug)
            ->where('status', 'published')
            ->with(['category', 'user', 'tags', 'gallery'])
            ->firstOrFail();

        // Incrementar visualizações
        $news->increment('views');

        // Mais lidas
        $mostRead = News::where('status', 'published')
            ->where('id', '!=', $news->id)
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();

        // Últimos posts
        $latestPosts = News::where('status', 'published')
            ->where('id', '!=', $news->id)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->limit(5)
            ->get();

        // Anúncios
        $sidebarAds = $this->getSidebarAds();

        return view('frontend.v2.news.show', compact(
            'news',
            'mostRead',
            'latestPosts',
            'sidebarAds'
        ));
    }

    /**
     * Listar notícias por categoria - V1
     */
    public function categoryV1($slug)
    {
        $category = Category::where('slug', $slug)
            ->where('status', 'active')
            ->firstOrFail();

        $news = News::where('status', 'published')
            ->where('category_id', $category->id)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->paginate(12);

        $sidebarAds = $this->getSidebarAds();

        return view('frontend.v1.category', compact('category', 'news', 'sidebarAds'));
    }

    /**
     * Listar notícias por categoria - V2
     */
    public function categoryV2($slug)
    {
        $category = Category::where('slug', $slug)
            ->where('status', 'active')
            ->firstOrFail();

        $news = News::where('status', 'published')
            ->where('category_id', $category->id)
            ->with(['category', 'user'])
            ->latest('published_at')
            ->paginate(10);

        $sidebarAds = $this->getSidebarAds();

        return view('frontend.v2.category', compact('category', 'news', 'sidebarAds'));
    }

    /**
     * Buscar notícias
     */
    public function search(Request $request)
    {
        $query = $request->get('q', '');
        $template = $request->get('template', 'v1'); // v1 ou v2

        if (empty($query)) {
            return redirect()->route('home.' . $template);
        }

        $news = News::where('status', 'published')
            ->where(function($q) use ($query) {
                $q->where('title', 'like', "%{$query}%")
                  ->orWhere('body', 'like', "%{$query}%")
                  ->orWhere('excerpt', 'like', "%{$query}%");
            })
            ->with(['category', 'user'])
            ->latest('published_at')
            ->paginate(12);

        $sidebarAds = $this->getSidebarAds();

        return view('frontend.' . $template . '.search', compact('news', 'query', 'sidebarAds'));
    }

    /**
     * Obter anúncios para sidebar
     */
    protected function getSidebarAds()
    {
        return [
            'vertical_300x600' => \App\Models\AdCampaign::whereHas('position', function($q) {
                    $q->where('slug', 'arranha-ceu-300x600');
                })
                ->active()
                ->inRandomOrder()
                ->first(),
            'rectangle_300x250' => \App\Models\AdCampaign::whereHas('position', function($q) {
                    $q->where('slug', 'retangulo-300x250');
                })
                ->active()
                ->inRandomOrder()
                ->first(),
        ];
    }
}

