<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use App\Services\MasterPanelService;
use Symfony\Component\HttpFoundation\Response;

class LicenseCheck
{
    protected $masterPanel;

    public function __construct(MasterPanelService $masterPanel)
    {
        $this->masterPanel = $masterPanel;
    }

    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Verificar se a validação de licença está desabilitada
        if (config('master-panel.disable_license_check', false)) {
            \Log::info('License check disabled - allowing request');
            return $next($request);
        }

        // Pular validação em rotas de instalação e login
        if ($request->is('install*') || 
            $request->is('api/install*') || 
            $request->is('login*') ||
            $request->is('setup*') ||
            $request->is('installer*')) {
            return $next($request);
        }

        // Verificar se o sistema já foi instalado
        if (!file_exists(storage_path('installed'))) {
            // Se não foi instalado, permitir acesso para instalação
            return $next($request);
        }

        // Verificar se há chave de licença configurada
        $licenseKey = config('master-panel.license_key');
        if (empty($licenseKey)) {
            // Se não há licença configurada, permitir acesso (modo desenvolvimento)
            // Mas mostrar aviso se for produção
            if (config('app.env') === 'production') {
                if ($request->expectsJson()) {
                    return response()->json([
                        'error' => 'Licença não configurada',
                        'message' => 'Configure a chave de licença no arquivo .env'
                    ], 403);
                }

                return response()->view('errors.license', [
                    'message' => 'Licença não configurada. Configure MASTER_PANEL_LICENSE_KEY no arquivo .env'
                ], 403);
            }
            
            // Em desenvolvimento, permitir sem licença
            return $next($request);
        }

        // Validar licença
        try {
            $ip = $request->ip();
            $domain = $request->getHost();
            
            \Log::info('Validating license', [
                'ip' => $ip,
                'domain' => $domain,
                'license_key' => substr($licenseKey, 0, 10) . '...',
                'master_url' => config('master-panel.url')
            ]);
            
            $validation = $this->masterPanel->validateLicense($ip, $domain);
            
            \Log::info('License validation result', [
                'success' => $validation['success'] ?? false,
                'message' => $validation['message'] ?? 'N/A'
            ]);
            
            if (!isset($validation['success']) || !$validation['success']) {
                $message = $validation['message'] ?? 'Erro ao validar licença. Verifique a URL do Master Panel e sua conexão com a internet.';
                
                // Em desenvolvimento, permitir mesmo com erro
                if (config('app.env') !== 'production') {
                    \Log::warning('License validation failed but allowing in development', [
                        'message' => $message
                    ]);
                    return $next($request);
                }
                
                if ($request->expectsJson()) {
                    return response()->json([
                        'error' => 'Licença inválida ou expirada',
                        'message' => $message,
                        'debug' => [
                            'url' => config('master-panel.url'),
                            'has_license_key' => !empty($licenseKey)
                        ]
                    ], 403);
                }

                return response()->view('errors.license', [
                    'message' => $message,
                    'debug' => config('app.debug') ? [
                        'url' => config('master-panel.url'),
                        'has_license_key' => !empty($licenseKey)
                    ] : null
                ], 403);
            }
        } catch (\Exception $e) {
            \Log::error('License validation exception', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Em caso de erro de conexão, permitir acesso em desenvolvimento
            if (config('app.env') !== 'production') {
                \Log::warning('Erro ao validar licença (modo desenvolvimento): ' . $e->getMessage());
                return $next($request);
            }

            // Em produção, bloquear se não conseguir validar
            $errorMessage = 'Erro ao validar licença. Verifique a URL do Master Panel (' . config('master-panel.url', 'master.canal33.com.br') . ') e sua conexão com a internet. Erro: ' . $e->getMessage();
            
            if ($request->expectsJson()) {
                return response()->json([
                    'error' => 'Erro ao validar licença',
                    'message' => $errorMessage,
                    'debug' => config('app.debug') ? [
                        'exception' => get_class($e),
                        'file' => $e->getFile(),
                        'line' => $e->getLine()
                    ] : null
                ], 503);
            }

            return response()->view('errors.license', [
                'message' => $errorMessage,
                'debug' => config('app.debug') ? [
                    'exception' => get_class($e),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ] : null
            ], 503);
        }

        return $next($request);
    }
}

