<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected $apiUrl;
    protected $name;
    protected $token;

    public function __construct()
    {
        $this->apiUrl = config('whatsapp.api_url');
        $this->name = config('whatsapp.name');
        $this->token = config('whatsapp.token');
    }

    /**
     * Enviar mensagem WhatsApp
     */
    public function sendMessage($number, $message)
    {
        try {
            $number = $this->formatNumber($number);
            
            $url = $this->apiUrl . '?' . http_build_query([
                'name' => $this->name,
                'token' => $this->token,
                'number' => $number,
                'message' => $message
            ]);

            $response = Http::timeout(30)->get($url);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'response' => $response->body()
                ];
            }

            return [
                'success' => false,
                'error' => 'Erro ao enviar mensagem'
            ];
        } catch (\Exception $e) {
            Log::error('WhatsApp send error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Formatar número de telefone
     */
    protected function formatNumber($number)
    {
        $number = preg_replace('/[^0-9]/', '', $number);
        
        if (substr($number, 0, 1) === '0') {
            $number = substr($number, 1);
        }

        if (strlen($number) <= 11 && substr($number, 0, 2) !== '55') {
            $number = '55' . $number;
        }

        return $number;
    }

    /**
     * Enviar notificação de nova notícia
     */
    public function notifyNewNews($number, $newsTitle, $newsUrl)
    {
        $message = "📰 Nova Notícia Publicada!\n\n";
        $message .= "Título: {$newsTitle}\n";
        $message .= "Acesse: {$newsUrl}";
        
        return $this->sendMessage($number, $message);
    }

    /**
     * Enviar notificação de atualização
     */
    public function notifyUpdate($number, $version, $title)
    {
        $message = "🚀 Nova Atualização Disponível!\n\n";
        $message .= "Versão: {$version}\n";
        $message .= "Título: {$title}\n";
        $message .= "\nAcesse seu painel para atualizar.";
        
        return $this->sendMessage($number, $message);
    }
}

