@extends('admin.layouts.app')

@section('title', 'Gerir Categorias')

@section('content')
<div class="categories-container">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2>Gerir Categorias</h2>
        <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#categoryModal" onclick="openCategoryModal()">
            <i class="bi bi-plus-circle"></i> Nova Categoria
        </button>
    </div>

    <!-- Filtros -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" action="{{ route('admin.categories.index') }}" class="row g-3">
                <div class="col-md-4">
                    <input type="text" name="search" class="form-control" 
                           value="{{ request('search') }}" 
                           placeholder="Buscar categoria...">
                </div>
                <div class="col-md-2">
                    <select name="status" class="form-select">
                        <option value="">Todos</option>
                        <option value="active" {{ request('status') == 'active' ? 'selected' : '' }}>Ativos</option>
                        <option value="inactive" {{ request('status') == 'inactive' ? 'selected' : '' }}>Inativos</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="bi bi-search"></i> Filtrar
                    </button>
                </div>
                <div class="col-md-2">
                    <a href="{{ route('admin.categories.index') }}" class="btn btn-secondary w-100">
                        <i class="bi bi-x-circle"></i> Limpar
                    </a>
                </div>
            </form>
        </div>
    </div>

    <!-- Lista de Categorias -->
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0">Categorias ({{ $categories->count() }})</h5>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-hover">
                    <thead>
                        <tr>
                            <th width="50">Ordem</th>
                            <th>Nome</th>
                            <th>Slug/Link</th>
                            <th>Descrição</th>
                            <th width="100">Cor</th>
                            <th width="100">Status</th>
                            <th width="150">Ações</th>
                        </tr>
                    </thead>
                    <tbody id="sortableCategories">
                        @forelse($categories as $category)
                        <tr data-id="{{ $category->id }}">
                            <td>
                                <span class="badge badge-info">{{ $category->order ?? 0 }}</span>
                            </td>
                            <td>
                                <strong>{{ $category->name }}</strong>
                            </td>
                            <td>
                                <code>{{ $category->slug }}</code>
                            </td>
                            <td>
                                <small class="text-muted">{{ Str::limit($category->description, 50) }}</small>
                            </td>
                            <td>
                                <div class="d-flex align-items-center gap-2">
                                    <div style="width: 30px; height: 30px; background: {{ $category->color ?? '#667eea' }}; border-radius: 4px; border: 1px solid #ddd;"></div>
                                    <small>{{ $category->color ?? '#667eea' }}</small>
                                </div>
                            </td>
                            <td>
                                @if($category->status === 'active')
                                    <span class="badge badge-success">Ativo</span>
                                @else
                                    <span class="badge badge-secondary">Inativo</span>
                                @endif
                            </td>
                            <td>
                                <div class="btn-group btn-group-sm">
                                    <button type="button" 
                                            class="btn btn-outline-primary" 
                                            onclick="openCategoryModal({{ $category->id }})"
                                            title="Editar">
                                        <i class="bi bi-pencil"></i>
                                    </button>
                                    <button type="button" 
                                            class="btn btn-outline-{{ $category->status === 'active' ? 'warning' : 'success' }}" 
                                            onclick="toggleStatus({{ $category->id }}, '{{ $category->status }}')"
                                            title="{{ $category->status === 'active' ? 'Desativar' : 'Ativar' }}">
                                        <i class="bi bi-{{ $category->status === 'active' ? 'eye-slash' : 'eye' }}"></i>
                                    </button>
                                    <button type="button" 
                                            class="btn btn-outline-danger" 
                                            onclick="deleteCategory({{ $category->id }})"
                                            title="Excluir">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                </div>
                            </td>
                        </tr>
                        @empty
                        <tr>
                            <td colspan="7" class="text-center py-4">
                                <p class="text-muted mb-0">Nenhuma categoria encontrada</p>
                            </td>
                        </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Modal Categoria -->
<div class="modal fade" id="categoryModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="categoryModalTitle">Nova Categoria</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form id="categoryForm" method="POST">
                @csrf
                <div id="formMethod"></div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Nome <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control" required id="categoryName">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Slug/Link</label>
                        <input type="text" name="slug" class="form-control" id="categorySlug" readonly>
                        <small class="text-muted">Gerado automaticamente a partir do nome</small>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Descrição</label>
                        <textarea name="description" class="form-control" rows="3" id="categoryDescription"></textarea>
                    </div>
                    <div class="row">
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Cor <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <input type="color" name="color" class="form-control form-control-color" 
                                           value="#667eea" id="categoryColor" required>
                                    <input type="text" class="form-control" id="colorHex" value="#667eea" 
                                           pattern="^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$" maxlength="7">
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="mb-3">
                                <label class="form-label">Ordem</label>
                                <input type="number" name="order" class="form-control" value="0" min="0" id="categoryOrder">
                                <small class="text-muted">Número para ordenação (menor = primeiro)</small>
                            </div>
                        </div>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Status</label>
                        <select name="status" class="form-select" required id="categoryStatus">
                            <option value="active">Ativo</option>
                            <option value="inactive">Inativo</option>
                        </select>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Salvar</button>
                </div>
            </form>
        </div>
    </div>
</div>

@endsection

@push('styles')
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/jquery-ui-dist@1.13.2/jquery-ui.min.css">
<style>
    .ui-sortable-helper {
        background: #fff;
        box-shadow: 0 2px 8px rgba(0,0,0,0.15);
    }
    .form-control-color {
        width: 60px;
        height: 38px;
    }
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/jquery-ui-dist@1.13.2/jquery-ui.min.js"></script>
<script>
    let editingCategoryId = null;

    // Abrir modal
    function openCategoryModal(categoryId = null) {
        editingCategoryId = categoryId;
        const modal = document.getElementById('categoryModal');
        const form = document.getElementById('categoryForm');
        const title = document.getElementById('categoryModalTitle');
        const methodInput = document.getElementById('formMethod');

        if (categoryId) {
            title.textContent = 'Editar Categoria';
            form.action = `{{ route('admin.categories.index') }}/${categoryId}`;
            methodInput.innerHTML = '@method("PUT")';
            
            // Carregar dados da categoria
            fetch(`{{ route('admin.categories.index') }}/${categoryId}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('categoryName').value = data.name;
                    document.getElementById('categorySlug').value = data.slug;
                    document.getElementById('categoryDescription').value = data.description || '';
                    document.getElementById('categoryColor').value = data.color || '#667eea';
                    document.getElementById('colorHex').value = data.color || '#667eea';
                    document.getElementById('categoryOrder').value = data.order || 0;
                    document.getElementById('categoryStatus').value = data.status;
                });
        } else {
            title.textContent = 'Nova Categoria';
            form.action = '{{ route('admin.categories.store') }}';
            methodInput.innerHTML = '';
            form.reset();
            document.getElementById('categoryColor').value = '#667eea';
            document.getElementById('colorHex').value = '#667eea';
        }
    }

    // Sincronizar color picker com input de texto
    document.getElementById('categoryColor').addEventListener('input', function() {
        document.getElementById('colorHex').value = this.value;
    });

    document.getElementById('colorHex').addEventListener('input', function() {
        if (/^#[0-9A-F]{6}$/i.test(this.value)) {
            document.getElementById('categoryColor').value = this.value;
        }
    });

    // Gerar slug automaticamente
    document.getElementById('categoryName').addEventListener('input', function() {
        if (!editingCategoryId) {
            const slug = this.value.toLowerCase()
                .normalize('NFD')
                .replace(/[\u0300-\u036f]/g, '')
                .replace(/[^a-z0-9]+/g, '-')
                .replace(/(^-|-$)/g, '');
            document.getElementById('categorySlug').value = slug;
        }
    });

    // Toggle status
    function toggleStatus(id, currentStatus) {
        const newStatus = currentStatus === 'active' ? 'inactive' : 'active';
        fetch(`{{ route('admin.categories.index') }}/${id}`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
            },
            body: JSON.stringify({ status: newStatus })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                location.reload();
            }
        });
    }

    // Excluir categoria
    function deleteCategory(id) {
        if (confirm('Deseja realmente excluir esta categoria?')) {
            fetch(`{{ route('admin.categories.index') }}/${id}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                }
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert(data.message || 'Erro ao excluir categoria');
                }
            });
        }
    }

    // Ordenação drag and drop
    $(function() {
        $("#sortableCategories").sortable({
            handle: 'td:first-child',
            update: function(event, ui) {
                const items = [];
                $(this).children().each(function(index) {
                    items.push({
                        id: $(this).data('id'),
                        order: index + 1
                    });
                });

                fetch('{{ route('admin.categories.index') }}/reorder', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    },
                    body: JSON.stringify({ items })
                });
            }
        });
    });
</script>
@endpush

